using System;
using System.IO;
using DynaPDF;

namespace metafiles
{
	class Metafiles
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      static void PlaceEMFCentered(CPDF PDF, String FileName, double Width, double Height)
      {
         TRectL r = new TRectL();
         double x, y;
         /*
            We place the EMF file horizontal and vertically centered in this example.
         */
         // We need the picture size of the EMF file
         if (!PDF.GetLogMetafileSize(FileName, ref r)) return;

         double w = r.Right - r.Left;
         double h = r.Bottom - r.Top;
         /*
            We place the EMF file into a border of 2 units around the page so that we can better see
            the bounding box of the EMF file.
         */
         Width  -= 4.0;
         Height -= 4.0;
         double sx = Width / w;
         /*
            An important note about the bounding rectangle: DynaPDF calculates the zero point
            of the EMF picture automatically so that we don't need to consider the coordinate origin.
            The coordinate origin for our calculation here is always 0.0, 0.0 independent of the real
            origin of the EMF picture.
         */
         if (h * sx <= Height)
         {
            h *= sx;
            x = 2.0;
            // If the file should not be centered vertically then set y to 2.0.
            y = (Height - h) / 2.0;
            PDF.InsertMetafile(FileName, x, y, Width, 0.0);
            // The rectangle represents the real bounding box of the EMF picture.
            PDF.SetStrokeColor(CPDF.PDF_RED);
            PDF.Rectangle(x, y, Width, h, DynaPDF.TPathFillMode.fmStroke);
         }else
         {
            sx = Height / h;
            w *= sx;
            x = (Width - w) / 2.0;
            y = 2.0;
            PDF.InsertMetafile(FileName, x, y, 0.0, Height);
            // The rectangle represents the real bounding box of the EMF picture.
            PDF.SetStrokeColor(CPDF.PDF_RED);
            PDF.Rectangle(x, y, w, Height, DynaPDF.TPathFillMode.fmStroke);
         }
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new DynaPDF.TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We open the output file later if no error occurrs.

            pdf.SetDocInfo(DynaPDF.TDocumentInfo.diCreator, "C# test application");
            pdf.SetDocInfo(DynaPDF.TDocumentInfo.diTitle, "Metafiles");

            pdf.SetPageCoords(DynaPDF.TPageCoord.pcTopDown);

            pdf.Append();
            /*
               We use a landscape paper format in this example. SetOrientationEx() rotates the coordinate
               system according to the orientation and sets the page orientation. You can now work with page
               as if it was not rotated. The real page format is still DIN A4 (this is the default format).
               The difference to SetOrientation() is that this function does not change the page's coordinate
               system.

               It would also be possible to use a user defined paper format without changing the page
               orientation but the disadvantage is that a printer driver must then manually rotate the page
               because landscape paper formats do not exist in most printers. This step requires an
               additional interaction with the user which is simply not required when creating landscape
               paper formats in the right way.
            */
            pdf.SetOrientationEx(90);

            // This file transforms the coordinate system very often and uses clipping regions. The metafile
            // is scaled to the page width without changing its aspect ratio.
            PlaceEMFCentered(pdf, "../../../test_files/coords.emf", pdf.GetPageWidth(), pdf.GetPageHeight());
            pdf.EndPage();

            pdf.Append();
            pdf.SetOrientationEx(90);
            // Simple test of line and standard patterns
            PlaceEMFCentered(pdf, "../../../test_files/fulltest.emf", pdf.GetPageWidth(), pdf.GetPageHeight());
            pdf.EndPage();

            pdf.Append();
            pdf.SetOrientationEx(90);
            // Outlined text, dash patterns with text. This file requires the font Bookman Old Style. If not available,
            // the result will be wrong!
            PlaceEMFCentered(pdf, "../../../test_files/gdi.emf", pdf.GetPageWidth(), pdf.GetPageHeight());
            pdf.EndPage();

            // No fatal error occurred?
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
